//
//  FSMenuActionManager.m
//  Downloader
//
//  Created by LiDong on 13-3-15.
//  Copyright (c) 2013年 LiDong. All rights reserved.
//

#import "FSOperationManager.h"
#import "FSFileItem.h"

@implementation FSOperationManager

@synthesize operationQueue = _operationQueue;

static FSOperationManager *_sharedManager = nil;

+ (FSOperationManager *)sharedManager {
    if (nil == _sharedManager) {
        _sharedManager = [[FSOperationManager alloc] init];
    }
    return _sharedManager;
}

- (id)init {
    if (self = [super init]) {
        _operationQueue = [[NSOperationQueue alloc] init];
    }
    return self;
}

- (void)dealloc {
    [_operationQueue cancelAllOperations];
    [_activityView removeFromSuperview];
}

- (BOOL)canPaste {
    return [_fileOperation canPaste];
}

- (void)cancel {
    [_operationQueue cancelAllOperations];
}

- (BOOL)createOperationForType:(OPERATION_TYPE)type items:(NSArray *)fileItems handler:(FSCompletionHandler)block {
    if ([fileItems count] > 0) {
        [_operationQueue cancelAllOperations];
        _fileOperation = [[FSFileOperation alloc] initWithOperationType:type];
        [_fileOperation setDelegate:self];
        [_fileOperation setFileItems:fileItems];
        [_fileOperation setSrcCompletionHandler:block];
        return YES;
    }
    return NO;
}

- (void)copyFileItems:(NSArray *)fileItems completionBlock:(FSCompletionHandler)block {
    [self createOperationForType:OPERATION_COPY items:fileItems handler:block];
}

- (void)cutFileItems:(NSArray *)fileItems completionBlock:(FSCompletionHandler)block {
    [self createOperationForType:OPERATION_CUT items:fileItems handler:block];
}

- (void)removeFileItems:(NSArray *)fileItems completionBlock:(FSCompletionHandler)block {
    if ([self createOperationForType:OPERATION_REMOVE items:fileItems handler:block]) {
        [_operationQueue addOperation:_fileOperation];
    }
}

- (void)zipFileItems:(NSArray *)fileItems toPath:(NSString *)dstPath completionBlock:(FSCompletionHandler)block {
    if ([self createOperationForType:OPERATION_ZIP items:fileItems handler:block]) {
        [_fileOperation setDstPath:dstPath];
        [_operationQueue addOperation:_fileOperation];
    }
}

- (void)zipFileItems:(NSArray *)fileItems completionBlock:(FSCompletionHandler)block {
    if ([fileItems count] > 0) {
        NSString *dstPath = [[[fileItems objectAtIndex:0] path] stringByDeletingLastPathComponent];
        
        [self zipFileItems:fileItems toPath:dstPath completionBlock:block];
    }
}

- (void)unzipFileItem:(FSFileItem *)fileItem completionBlock:(FSCompletionHandler)block {
    NSArray *fileItems = [[NSArray alloc] initWithObjects:fileItem, nil];
    
    if ([self createOperationForType:OPERATION_UNZIP items:fileItems handler:block]) {
        [_fileOperation setDstPath:[fileItem path]];
        [_operationQueue addOperation:_fileOperation];
    }
    }

- (void)pasteToFileItem:(FSFileItem *)targetItem completionBlock:(FSCompletionHandler)block {
    if ([self canPaste]) {
        if (OPERATION_COPY == [_fileOperation operationType] && ([_fileOperation isFinished] || [_fileOperation isCancelled])) {
            FSFileOperation *newOperation = [_fileOperation copy];
            
            _fileOperation = newOperation;
        }
        [_fileOperation setDstPath:[targetItem path]];
        [_fileOperation setDstCompletionHandler:block];
        [_operationQueue addOperation:_fileOperation];
    }
}

#pragma mark - 

- (void)fileOperationDidBegin:(FSFileOperation *)operation {
    if (nil == _activityView) {
        NSString *title = nil;
        const OPERATION_TYPE type = [operation operationType];
        
        if (OPERATION_COPY == type) {
            title = LS(@"COPYING");
        } else if (OPERATION_CUT == type) {
            title = LS(@"CUTTING");
        } else if (OPERATION_REMOVE == type) {
            title = LS(@"REMOVING");
        } else if (OPERATION_ZIP == type) {
            title = LS(@"ARCHIVING");
        } else if (OPERATION_UNZIP == type) {
            title = LS(@"EXTRACTING");
        } else {
            return;
        }
        
        _activityView = [[IFActivityView alloc] initWithTitle:title];
        [_activityView setDelegate:self];
        [_activityView show];
    }
}

- (void)fileOperationDidEnd:(FSFileOperation *)operation {
    if (operation == _fileOperation) {
        if ([_fileOperation isCancelled]) {
            [_activityView abort];
        } else {
            [_activityView done];
        }
        
        if (OPERATION_COPY != [_fileOperation operationType]) {
            _fileOperation = nil;
        }
    }
}

#pragma mark -


- (void)activityViewDidDismiss:(IFActivityView *)activityView {
    if (activityView == _activityView) {
        _activityView  = nil;
    }
}

@end
